<?php

namespace App\Models;;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;

class Item extends Model
{
    protected $fillable = ['name', 'description'];



    public function trades()
    {
        return $this->hasMany(GoldTrade::class);
    }

    public function tests()
    {
        return $this->hasMany(GoldTest::class);
    }


    // stock, avgFinalUSD, totalAFN, totalUSD
    public static function stockSummary($id, $toDate)
    {
        $open = OpeningStock::where('item_id', $id)
            ->where('date', '<=', $toDate)
            ->sum('weight_gram');

        $purchase = GoldTrade::where('item_id', $id)
            ->where('type', 'purchase')
            ->whereDate('created_at', '<=', $toDate)
            ->sum('weight_gram');

        $sale = GoldTrade::where('item_id', $id)
            ->where('type', 'sale')
            ->whereDate('created_at', '<=', $toDate)
            ->sum('weight_gram');



        $rate = CurrencyRate::where('currency', 'AFN')
            ->whereDate('date', '<=', $toDate)
            ->orderByDesc('date')
            ->value('rate') ?? 1;

        $stock = $open + $purchase - $sale;
        $totals = GoldTrade::selectRaw("
                SUM(CASE WHEN currency = 'USD' THEN weight_gram * price_per_gram ELSE 0 END) AS usd_tot,
                SUM(CASE WHEN currency = 'AFN' THEN weight_gram * price_per_gram ELSE 0 END) AS afn_tot
            ")
            ->where('item_id', $id)
            ->where('type', 'purchase')
            ->whereDate('created_at', '<=', $toDate)
            ->first();

        $usd = $totals->usd_tot ?? 0;
        $afn = $totals->afn_tot ?? 0;

        $totals = OpeningStock::selectRaw("SUM(weight_gram * price_gram) AS usd_tot")
            ->where('item_id', $id)
            ->whereDate('date', '<=', $toDate)
            ->first();

        $usd += $totals->usd_tot ?? 0;

        $totalPrice = ($afn / $rate) + $usd;
        $avgUSD = $stock > 0 ? round($totalPrice / $stock, 2) : 0;



        return [
            $stock,
            round($avgUSD, 2),
            round($usd, 2),     // pure USD total price
            round($afn, 2),     // pure AFN total price
        ];
    }

    // AVG PRICE TO AFN OR USD
    public static function ItemAvgPrice($id, $toDate, $currency)
    {
        $currency = strtoupper($currency);
        if (!in_array($currency, ['AFN', 'USD'])) {
            return 0;
        }

        $open = OpeningStock::where('item_id', $id)
            ->where('date', '<=', $toDate)
            ->get();

        $trades = GoldTrade::where('item_id', $id)
            ->where('type', 'purchase')
            ->whereDate('created_at', '<=', $toDate)
            ->where('currency', $currency)
            ->get();

        $weight = 0;
        $total = 0;

        if ($currency === 'USD') {
            foreach ($open as $o) {
                $weight += $o->weight_gram;
                $total += $o->weight_gram * $o->price_gram;
            }
        }

        foreach ($trades as $t) {
            $weight += $t->weight_gram;
            $total += $t->weight_gram * $t->price_per_gram;
        }

        return $weight > 0 ? round($total / $weight, 2) : 0;
    }


    public static function ItemProfitSummary($id, $toDate)
    {
        $s = GoldTrade::where('type', 'sale')
            ->where('item_id', $id)
            ->whereDate('created_at', '<=', $toDate);

        $t = GoldTest::where('item_id', $id)
            ->whereDate('created_at', '<=', $toDate);

        $c = $s->get()->map(function ($r) {
            $cp = 10;
            $gp = ($r->price_per_gram - $cp) * $r->weight_gram;
            return [
                'cur' => $r->currency,
                'gp' => $gp,
            ];
        })->concat(
            $t->get()->map(function ($r) {
                return [
                    'cur' => $r->currency,
                    'gp' => $r->weight_gram * $r->rate_per_gram,
                ];
            })
        );

        $afn = $c->where('cur', 'AFN')->sum('gp');
        $usd = $c->where('cur', 'USD')->sum('gp');

        return [
            round($afn, 2),
            round($usd, 2),
        ];
    }
}
