<?php

namespace App\Filament\Widgets;

use App\Filament\ReportFuncs\AccountRetrievalReport;
use App\Models\CurrencyExchange;
use App\Models\CurrencyRate;
use App\Models\Expense;
use App\Models\Item;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;

class FinancialSummaryWidget extends BaseWidget
{
    protected bool $isCollapsible = true;

    protected function getStats(): array
    {
        $date = date('Y-m-d', strtotime('tomorrow'));
        $totalAssetsAfn = $totalAssetsUsd  = 0;


        $rate = CurrencyRate::where('currency', 'AFN')
            ->latest('date')
            ->value('rate') ?? 80;

        // transaction section
        {
            $data = AccountRetrievalReport::generate($date, null);
            $results = collect($data['results'] ?? []);

            // global assets of transaction section
            $totalAssetsAfn += $results->sum('balance_afn');
            $totalAssetsUsd += $results->sum('balance_usd');



            $cashResults = $results->filter(fn($a) => ($a['type'] ?? null) === 'cash');
            $NoCash = $results->filter(fn($a) => ($a['type'] ?? null) !== 'cash');

            $cashAfn = round($cashResults->sum('balance_afn'), 2);
            $cashUsd = round($cashResults->sum('balance_usd'), 2);

            $receivableAfn = round($NoCash->sum(fn($a) => max($a['balance_afn'] ?? 0, 0)), 2);
            $payableAfn = round($NoCash->sum(fn($a) => min($a['balance_afn'] ?? 0, 0)) * -1, 2);
            $receivableUsd = round($NoCash->sum(fn($a) => max($a['balance_usd'] ?? 0, 0)), 2);
            $payableUsd = round($NoCash->sum(fn($a) => min($a['balance_usd'] ?? 0, 0)) * -1, 2);
        }


        // start of stock values

        $stockAfn = $stockUsd = 0;

        foreach (Item::all() as $item) {
            [$x, $y, $usdTotal, $afnTotal] = Item::stockSummary($item->id, $date);

            $stockAfn += $afnTotal;
            $stockUsd += $usdTotal;
        }

        $stockAfn = round($stockAfn, 2);
        $stockUsd = round($stockUsd, 2);

        $totalAssetsAfn += $stockAfn;
        $totalAssetsUsd += $stockUsd;

        $profitUsd = 0;
        $profitAfn = 0;
        // calc profit
        {

            foreach (Item::all() as $item) {
                [$afn, $usd] = Item::ItemProfitSummary($item->id, $date);
                $profitAfn += $afn;
                $profitUsd += $usd;
            }

            CurrencyExchange::whereDate('created_at', '<=', $date)->get()->each(function ($r) use (&$profitAfn, &$profitUsd) {
                if ($r->profit_currency === 'AFN') {
                    $profitAfn += $r->profit_amount;
                } elseif ($r->profit_currency === 'USD') {
                    $profitUsd += $r->profit_amount;
                }
            });

            Expense::whereDate('date', '<=', $date)->get()->each(function ($e) use (&$profitAfn, &$profitUsd) {
                if ($e->currency === 'AFN') {
                    $profitAfn -= $e->amount;
                } elseif ($e->currency === 'USD') {
                    $profitUsd -= $e->amount;
                }
            });

            $profitUsd = round($profitUsd, 2);
            $profitAfn = round($profitAfn, 2);
        }

        return [
            Stat::make('Cash Balance', '$' . number_format($cashUsd + $cashAfn / $rate, 2))
                ->description(number_format($cashAfn) . ' AFN | $' . number_format($cashUsd))
                ->icon('heroicon-o-banknotes')
                ->color('success'),

            Stat::make('Accounts Receivable', '$' . number_format($receivableUsd + $receivableAfn / $rate, 2))
                ->description(number_format($receivableAfn) . ' AFN | $' . number_format($receivableUsd))
                ->icon('heroicon-o-arrow-down-circle')
                ->color('warning'),

            Stat::make('Accounts Payable', '$' . number_format($payableUsd + $payableAfn / $rate, 2))
                ->description(number_format($payableAfn) . ' AFN | $' . number_format($payableUsd))
                ->icon('heroicon-o-arrow-up-circle')
                ->color('danger'),

            Stat::make('stock Value', '$' . number_format($stockUsd + $stockAfn / $rate, 2))
                ->description(number_format($stockAfn) . ' AFN | $' . number_format($stockUsd))
                ->icon('heroicon-o-cube')
                ->color('primary'),

            Stat::make('Total Assets', '$' . number_format($totalAssetsUsd + $totalAssetsAfn / $rate, 2))
                ->description(number_format($totalAssetsAfn) . ' AFN | $' . number_format($totalAssetsUsd))
                ->icon('heroicon-o-scale')
                ->color('info'),

            Stat::make('Net Profit', '$' . number_format($profitUsd + $profitAfn / $rate, 2))
                ->description(number_format($profitAfn) . ' AFN | $' . number_format($profitUsd))
                ->icon('heroicon-o-chart-bar')
                ->color('success'),

            Stat::make('Exchange Rate', '1 USD = ' . number_format($rate, 2) . ' AFN')
                ->description('Current conversion rate')
                ->icon('heroicon-o-arrows-right-left')
                ->color('gray'),
        ];
    }

    protected function getColumns(): int
    {
        return 3;
    }
}
