<?php

namespace App\Filament\Resources;

use App\Filament\Resources\GoldTradeResource\Pages;
use App\Models\GoldTrade;
use Filament\Forms;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Form;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;

class GoldTradeResource extends Resource
{
    protected static ?string $model = GoldTrade::class;

    protected static ?string $navigationIcon = 'heroicon-o-scale';

    protected static ?string $navigationGroup = 'Gold Operations';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make(3)->schema([
                    Select::make('type')->searchable()->preload()
                        ->options(['purchase' => 'Purchase', 'sale' => 'Sale'])->required(),
                    Select::make('account_id')->searchable()->preload()
                        ->relationship('account', 'name')->required(),
                    Select::make('item_id')->searchable()->preload()
                        ->relationship('item', 'name')->required(),
                    TextInput::make('weight_gram')
                        ->autocomplete('off')
                        ->numeric()
                        ->required()
                        ->reactive()
                        ->afterStateUpdated(
                            fn($state, callable $set, $get) =>
                            $set('total_price', $state * $get('price_per_gram'))
                        ),
                    TextInput::make('price_per_gram')
                        ->autocomplete('off')
                        ->numeric()
                        ->required()
                        ->reactive()
                        ->afterStateUpdated(
                            fn($state, callable $set, $get) =>
                            $set('total_price', $state * $get('weight_gram'))
                        ),
                    TextInput::make('total_price')
                        ->autocomplete('off')
                        ->numeric()
                        ->disabled()
                        ->dehydrated(false)
                        ->afterStateHydrated(
                            fn(callable $set, callable $get) =>
                            $set('total_price', $get('weight_gram') * $get('price_per_gram'))
                        ),
                    Select::make('currency')->searchable()->preload()
                        ->options(['AFN' => 'AFN', 'USD' => 'USD'])->required(),
                    DatePicker::make('created_at')->default(now())->required(),
                ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('id')->sortable(),
                TextColumn::make('type')->sortable(),
                TextColumn::make('account.name')->label('Account')->sortable()->searchable(),
                TextColumn::make('item.name')->label('Item')->sortable()->searchable(),
                TextColumn::make('weight_gram')->sortable(),
                TextColumn::make('price_per_gram')->sortable(),
                TextColumn::make('total_price')
                    ->label('Total Price')
                    ->getStateUsing(
                        fn(GoldTrade $record) =>
                        $record->weight_gram * $record->price_per_gram
                    )
                    ->sortable(false),
                TextColumn::make('currency')->sortable(),
                TextColumn::make('created_at')->date('Y-m-d')->sortable(),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\ViewAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListGoldTrades::route('/'),
            'create' => Pages\CreateGoldTrade::route('/create'),
            'edit' => Pages\EditGoldTrade::route('/{record}/edit'),
        ];
    }
}
