<?php

namespace App\Filament\Resources;

use App\Filament\Resources\GoldTestResource\Pages;
use App\Models\GoldTest;
use Filament\Forms\Form;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;

class GoldTestResource extends Resource
{
    protected static ?string $model = GoldTest::class;

    protected static ?string $navigationIcon = 'heroicon-o-beaker';

    protected static ?string $navigationGroup = 'Gold Operations';

    protected static ?int $navigationSort = 3;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make(3)->schema([
                    Select::make('customer_id')
                        ->searchable()
                        ->relationship('customer', 'name')
                        ->required()
                        ->preload(),
                    Select::make('item_id')
                        ->searchable()
                        ->relationship('item', 'name')
                        ->required()->preload(),
                    TextInput::make('weight_gram')
                        ->autocomplete('off')
                        ->numeric()
                        ->default(1)
                        ->required()
                        ->reactive()
                        ->afterStateUpdated(
                            fn($state, callable $set, $get) =>
                            $set('total_fee', $state * $get('rate_per_gram'))
                        ),
                    TextInput::make('rate_per_gram')
                        ->autocomplete('off')
                        ->numeric()
                        ->default(100)
                        ->required()
                        ->reactive()
                        ->afterStateUpdated(
                            fn($state, callable $set, $get) =>
                            $set('total_fee', $state * $get('weight_gram'))
                        ),
                    TextInput::make('total_fee')
                        ->autocomplete('off')
                        ->numeric()
                        ->disabled()
                        ->dehydrated(false)
                        ->afterStateHydrated(
                            fn(callable $set, callable $get) =>
                            $set('total_fee', $get('weight_gram') * $get('rate_per_gram'))
                        ),
                    Select::make('currency')->searchable()->preload()
                        ->options(['AFN' => 'AFN', 'USD' => 'USD'])->required(),
                    DatePicker::make('created_at')->default(now())->required(),
                ])

            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('id')->sortable(),
                TextColumn::make('customer.name')->label('Customer')->sortable()->searchable(),
                TextColumn::make('item.name')->label('Item')->sortable()->searchable(),
                TextColumn::make('weight_gram')->sortable(),
                TextColumn::make('rate_per_gram')->sortable(),
                TextColumn::make('total_fee')
                    ->label('Total Fee')
                    ->getStateUsing(
                        fn(GoldTest $record) =>
                        $record->weight_gram * $record->rate_per_gram
                    )
                    ->sortable(false),
                TextColumn::make('currency')->sortable(),
                TextColumn::make('created_at')->date('Y-m-d')->sortable(),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\ViewAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListGoldTests::route('/'),
            'create' => Pages\CreateGoldTest::route('/create'),
            'edit' => Pages\EditGoldTest::route('/{record}/edit'),
        ];
    }
}
