<?php

namespace App\Filament\Resources;

use App\Filament\Resources\CurrencyExchangeResource\Pages;

use App\Models\CurrencyExchange;
use App\Models\CurrencyRate;

use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Fieldset;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Tables\Columns\TextColumn;
use Filament\Forms\Components\Section;


class CurrencyExchangeResource extends Resource
{
    protected static ?string $model = CurrencyExchange::class;

    protected static ?string $navigationIcon = 'heroicon-o-currency-dollar';

    protected static ?string $navigationGroup = 'Financial Section';


    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Accounts & Rate')->schema([
                    Grid::make(3)->schema([
                        Select::make('source_account_id')
                            ->label('Source Account')
                            ->searchable()
                            ->preload()
                            ->relationship('sourceAccount', 'name', fn($query) => $query->whereIn('type', ['cash', 'bank']))
                            ->required(),

                        Select::make('exchanger_account_id')
                            ->label('Exchanger Account')
                            ->searchable()
                            ->preload()
                            ->relationship('exchangerAccount', 'name', fn($query) => $query->whereIn('type', ['cash', 'bank']))
                            ->required(),

                        TextInput::make('rate')
                            ->label('Exchange Rate')
                            ->numeric()
                            ->default(CurrencyRate::where('currency', 'AFN')->latest('date')->value('rate') ?? 80)
                            ->required(),
                    ]),
                ])->columns(1),

                Section::make('Exchange Details')->schema([
                    Grid::make(2)->schema([
                        Fieldset::make('Purchase')->schema([
                            TextInput::make('purchase_amount')->label('Amount')->numeric()->required(),
                            Select::make('purchase_currency')->label('Currency')->options(['AFN' => 'AFN', 'USD' => 'USD'])->searchable()->required(),
                        ])->columns(2)->columnSpan(1),

                        Fieldset::make('Sale')->schema([
                            TextInput::make('sale_amount')->label('Amount')->numeric()->required(),
                            Select::make('sale_currency')->label('Currency')->options(['AFN' => 'AFN', 'USD' => 'USD'])->searchable()->required(),
                        ])->columns(2)->columnSpan(1),
                    ]),
                ]),

                Section::make('Profit')->schema([
                    Grid::make(2)->schema([
                        TextInput::make('profit_amount')->label('Amount')->numeric()->required(),
                        Select::make('profit_currency')->label('Currency')->options(['AFN' => 'AFN', 'USD' => 'USD'])->searchable()->required(),
                    ]),
                ]),

                Section::make('Additional Info')->schema([
                    Grid::make(12)->schema([
                        Textarea::make('description')->label('Description')->columnSpan(9),
                        DatePicker::make('created_at')->label('Date')->default(now())->required()->columnSpan(3),
                    ]),
                ]),
            ]);
    }




    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('id')->label('#')->sortable(),

                TextColumn::make('sourceAccount.name')->label('Source Account')->sortable()->searchable(),

                TextColumn::make('exchangerAccount.name')->label('Exchanger Account')->sortable()->searchable(),

                TextColumn::make('rate')->label('Rate')->sortable(),

                TextColumn::make('purchase_amount')
                    ->label('Purchase Amount')
                    ->formatStateUsing(fn($state, $record) => number_format($state, 2) . ' ' . $record->purchase_currency)
                    ->sortable(),

                TextColumn::make('sale_amount')
                    ->label('Sale Amount')
                    ->formatStateUsing(fn($state, $record) => number_format($state, 2) . ' ' . $record->sale_currency)
                    ->sortable(),

                TextColumn::make('profit_amount')
                    ->label('Profit Amount')
                    ->formatStateUsing(fn($state, $record) => number_format($state, 2) . ' ' . $record->sale_currency)
                    ->sortable(),

                TextColumn::make('created_at')->label('Date')->date('Y-m-d')->sortable(),

                TextColumn::make('purchase_currency')
                    ->label('Purchase Currency')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('sale_currency')
                    ->label('Sale Currency')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('profit_currency')
                    ->label('Profit Currency')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('description')
                    ->label('Description')
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\ViewAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }


    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListCurrencyExchanges::route('/'),
            'create' => Pages\CreateCurrencyExchange::route('/create'),
            'edit' => Pages\EditCurrencyExchange::route('/{record}/edit'),
        ];
    }
}
