<?php

namespace App\Filament\ReportFuncs;

use App\Models\Account;
use App\Models\OpeningBalance;
use App\Models\CurrencyExchange;
use App\Models\Transaction;
use Carbon\Carbon;

class JournalReport
{
    public static function generate($from, $to): array
    {
        $acc_ids = Account::pluck('id')->toArray();

        $fromDate = Carbon::parse($from)->startOfDay();
        $toDate = Carbon::parse($to)->endOfDay();

        $entries = [];

        foreach (OpeningBalance::whereBetween('date', [$fromDate, $toDate])->get() as $r) {
            $acc = Account::find($r->account_id)?->name ?? $r->account_id;
            $sys = 'System_Opening_Assets';
            $amt = round($r->amount, 2);

            $entries[] = [
                'no' => null,
                'date' => $r->date,
                'account_name' => $acc,
                'description' => 'Opening Balance',
                'afn_debit' => $r->currency === 'AFN' ? $amt : '',
                'afn_credit' => '',
                'usd_debit' => $r->currency === 'USD' ? $amt : '',
                'usd_credit' => '',
            ];

            $entries[] = [
                'no' => null,
                'date' => $r->date,
                'account_name' => $sys,
                'description' => 'Opening Balance',
                'afn_debit' => '',
                'afn_credit' => $r->currency === 'AFN' ? $amt : '',
                'usd_debit' => '',
                'usd_credit' => $r->currency === 'USD' ? $amt : '',
            ];
        }


        // Transactions (two rows: from = credit, to = debit)
        foreach (
            Transaction::where(function ($q) use ($acc_ids) {
                $q->whereIn('from_account_id', $acc_ids)->orWhereIn('to_account_id', $acc_ids);
            })->whereBetween('created_at', [$fromDate, $toDate])->get() as $r
        ) {
            $from = Account::find($r->from_account_id)?->name ?? $r->from_account_id;
            $to = Account::find($r->to_account_id)?->name ?? $r->to_account_id;

            $amt = round($r->amount, 2);


            $entries[] = [
                'no' => null,
                'date' => $r->created_at,
                'account_name' => $from,
                'description' => $r->from_note ?? "Transaction",
                'afn_debit' => '',
                'afn_credit' => $r->currency === 'AFN' ? $amt : '',
                'usd_debit' => '',
                'usd_credit' => $r->currency === 'USD' ? $amt : '',
            ];

            $entries[] = [
                'no' => null,
                'date' => $r->created_at,
                'account_name' => $to,
                'description' => $r->to_note ?? "Transaction",
                'afn_debit' => $r->currency === 'AFN' ? $amt : '',
                'afn_credit' => '',
                'usd_debit' => $r->currency === 'USD' ? $amt : '',
                'usd_credit' => '',
            ];
        }

        // Currency Exchange
        foreach (CurrencyExchange::whereBetween('created_at', [$fromDate, $toDate])->get() as $r) {
            $src = Account::find($r->source_account_id)?->name ?? $r->source_account_id;
            $ex = Account::find($r->exchanger_account_id)?->name ?? $r->exchanger_account_id;
            $desc = $r->description ?? 'Currency Exchange';

            $sale = round($r->sale_amount, 2);
            $buy = round($r->purchase_amount, 2);

            // Source account: credit sale, debit purchase
            $entries[] = [
                'no' => null,
                'date' => $r->created_at,
                'account_name' => $src,
                'description' => $desc,
                'afn_debit' => $r->purchase_currency === 'AFN' ? $buy : '',
                'afn_credit' => $r->sale_currency === 'AFN' ? $sale : '',
                'usd_debit' => $r->purchase_currency === 'USD' ? $buy : '',
                'usd_credit' => $r->sale_currency === 'USD' ? $sale : '',
            ];

            // Exchanger account: debit sale, credit purchase (reverse)
            $entries[] = [
                'no' => null,
                'date' => $r->created_at,
                'account_name' => $ex,
                'description' => $desc,
                'afn_debit' => $r->sale_currency === 'AFN' ? $sale : '',
                'afn_credit' => $r->purchase_currency === 'AFN' ? $buy : '',
                'usd_debit' => $r->sale_currency === 'USD' ? $sale : '',
                'usd_credit' => $r->purchase_currency === 'USD' ? $buy : '',
            ];
        }


        usort($entries, fn($a, $b) => strtotime($a['date']) <=> strtotime($b['date']));
        foreach ($entries as $i => &$e) $e['no'] = $i + 1;

        return ['results' => $entries];
    }
}
