<?php

namespace App\Filament\ReportFuncs;

use App\Models\GoldTrade;
use App\Models\Item;
use App\Models\OpeningStock;
use Carbon\Carbon;

class ItemLedgerReport
{
    public static function generate($from, $to, $itemId = 1): array
    {
        $fromDate = Carbon::parse($from)->startOfDay();
        $toDate = Carbon::parse($to)->endOfDay();

        $openings = OpeningStock::where('item_id', $itemId)
            ->selectRaw("'opening' as type, date, weight_gram, null as account_id, created_at")
            ->get();

        $trades = GoldTrade::where('item_id', $itemId)
            ->selectRaw("type, created_at as date, weight_gram, account_id, created_at")
            ->get();

        $all = $openings->concat($trades);

        $oldRecords = $all->filter(fn($r) => Carbon::parse($r->date)->lt($fromDate))->sortBy('date');
        $allRecords = $all->filter(fn($r) => Carbon::parse($r->date)->between($fromDate, $toDate))->sortBy('date');

        $openingBalance = 0;
        foreach ($oldRecords as $r) {
            if ($r->type === 'opening') {
                $openingBalance += $r->weight_gram;
            } elseif ($r->type === 'purchase') {
                $openingBalance += $r->weight_gram;
            } elseif ($r->type === 'sale') {
                $openingBalance -= $r->weight_gram;
            }
        }

        $balance = $openingBalance;
        $results = [];

        $results[] = [
            'no' => 0,
            'date' => $fromDate->format('Y-m-d'),
            'title' => 'Old Item Balance',
            'debit' => '',
            'credit' => '',
            'balance' => number_format($balance, 2),
        ];

        foreach ($allRecords as $i => $r) {
            $debit = $credit = 0;
            $desc = '';

            if ($r->type === 'opening') {
                $debit = $r->weight_gram;
                $balance += $debit;
                $desc = 'Opening Stock';
            } elseif ($r->type === 'purchase') {
                $debit = $r->weight_gram;
                $balance += $debit;
                $acc = $r->account?->name ?? $r->account_id;
                $desc = "Purchase from $acc";
            } elseif ($r->type === 'sale') {
                $credit = $r->weight_gram;
                $balance -= $credit;
                $acc = $r->account?->name ?? $r->account_id;
                $desc = "Sale to $acc";
            }

            $results[] = [
                'no' => $i + 1,
                'date' => Carbon::parse($r->date)->format('Y-m-d'),
                'title' => $desc,
                'debit' => $debit ? number_format($debit, 2) : '',
                'credit' => $credit ? number_format($credit, 2) : '',
                'balance' => number_format($balance, 2),
            ];
        }

        return [
            'results' => $results,
            'opening' => $openingBalance,
        ];
    }
}
